<?php
/**
 * PayPal Standard - Payment Gateway
 * ENABLE AUTO-RETURN https://www.paypal.com/rs/cgi-bin/webscr?cmd=p/mer/express_return_summary-outside
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_PayPal_Standard' ) ) {

    class TC_Gateway_PayPal_Standard extends TC_Gateway_API {

        var $plugin_name = 'paypal_standard';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;
        var $currencies = [];
        var $locales = [];
        var $permanently_active = false;
        var $skip_payment_screen = true;
        var $gateway_url = '';
        var $business, $SandboxFlag, $returnURL, $cancelURL, $API_Endpoint, $version, $currency, $locale, $ignore_ipn_errors, $sandbox_flag;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
            // add_action('tc_system_info_server_environment_options', array($this, 'add_system_info_options'));
        }

        function init() {
            global $tc;

            $this->admin_name = __( 'PayPal Standard', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'PayPal', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/paypal-standard.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-paypal-standard.png', $this->plugin_name );

            $this->currency = $this->get_option( 'currency', 'USD' );
            $this->business = $this->get_option( 'email' );
            $this->locale = $this->get_option( 'locale', 'US' );
            $this->ignore_ipn_errors = $this->get_option( 'ignore_ipn_errors', 'no' );
            $this->sandbox_flag = $this->get_option( 'mode', 'sandbox' );

            $this->gateway_url = ( 'live' == $this->sandbox_flag )
                ? 'https://www.paypal.com/cgi-bin/webscr'
                : 'https://www.sandbox.paypal.com/cgi-bin/webscr';

            $this->currencies = apply_filters( 'tc_paypal_standard_currencies', [
                "AUD" => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                "BRL" => __( 'BRL - Brazilian Real', 'tickera-event-ticketing-system' ),
                "CAD" => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                "CZK" => __( 'CZK - Czech Koruna', 'tickera-event-ticketing-system' ),
                "DKK" => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                "EUR" => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                "HKD" => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                "HUF" => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                "ILS" => __( 'ILS - Israeli New Shekel', 'tickera-event-ticketing-system' ),
                "JPY" => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
                "MYR" => __( 'MYR - Malaysian Ringgit', 'tickera-event-ticketing-system' ),
                "MXN" => __( 'MXN - Mexican Peso', 'tickera-event-ticketing-system' ),
                "NOK" => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                "NZD" => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                "PHP" => __( 'PHP - Philippine Peso', 'tickera-event-ticketing-system' ),
                "PLN" => __( 'PLN - Polish Zloty', 'tickera-event-ticketing-system' ),
                "GBP" => __( 'GBP - Pound Sterling', 'tickera-event-ticketing-system' ),
                "RUB" => __( 'RUB - Russian Ruble', 'tickera-event-ticketing-system' ),
                "SGD" => __( 'SGD - Singapore Dollar', 'tickera-event-ticketing-system' ),
                "SEK" => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
                "CHF" => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                "TWD" => __( 'TWD - Taiwan New Dollar', 'tickera-event-ticketing-system' ),
                "TRY" => __( 'TRY - Turkish Lira', 'tickera-event-ticketing-system' ),
                "USD" => __( 'USD - U.S. Dollar', 'tickera-event-ticketing-system' ),
                "THB" => __( 'THB - Thai Baht', 'tickera-event-ticketing-system' ),
            ]);

            $locales = array(
                'AU' => __( 'Australia', 'tickera-event-ticketing-system' ),
                'AT' => __( 'Austria', 'tickera-event-ticketing-system' ),
                'BE' => __( 'Belgium', 'tickera-event-ticketing-system' ),
                'BR' => __( 'Brazil', 'tickera-event-ticketing-system' ),
                'CA' => __( 'Canada', 'tickera-event-ticketing-system' ),
                'CH' => __( 'Switzerland', 'tickera-event-ticketing-system' ),
                'CN' => __( 'China', 'tickera-event-ticketing-system' ),
                'DE' => __( 'Germany', 'tickera-event-ticketing-system' ),
                'ES' => __( 'Spain', 'tickera-event-ticketing-system' ),
                'SG' => __( 'Singapore', 'tickera-event-ticketing-system' ),
                'GB' => __( 'United Kingdom', 'tickera-event-ticketing-system' ),
                'FR' => __( 'France', 'tickera-event-ticketing-system' ),
                'IT' => __( 'Italy', 'tickera-event-ticketing-system' ),
                'MX' => __( 'Mexico', 'tickera-event-ticketing-system' ),
                'NL' => __( 'Netherlands', 'tickera-event-ticketing-system' ),
                'NZ' => __( 'New Zealand', 'tickera-event-ticketing-system' ),
                'PL' => __( 'Poland', 'tickera-event-ticketing-system' ),
                'PT' => __( 'Portugal', 'tickera-event-ticketing-system' ),
                'RU' => __( 'Russia', 'tickera-event-ticketing-system' ),
                'US' => __( 'United States', 'tickera-event-ticketing-system' ),
                'MY' => __( 'Malaysia', 'tickera-event-ticketing-system' ),
                'PH' => __( 'Philippines', 'tickera-event-ticketing-system' ),
                'da_DK' => __( 'Danish (for Denmark only)', 'tickera-event-ticketing-system' ),
                'he_IL' => __( 'Hebrew (all)', 'tickera-event-ticketing-system' ),
                'id_ID' => __( 'Indonesian (for Indonesia only)', 'tickera-event-ticketing-system' ),
                'ja_JP' => __( 'Japanese (for Japan only)', 'tickera-event-ticketing-system' ),
                'no_NO' => __( 'Norwegian (for Norway only)', 'tickera-event-ticketing-system' ),
                'pt_BR' => __( 'Brazilian Portuguese (for Portugal and Brazil only)', 'tickera-event-ticketing-system' ),
                'ru_RU' => __( 'Russian (for Lithuania, Latvia, and Ukraine only)', 'tickera-event-ticketing-system' ),
                'sv_SE' => __( 'Swedish (for Sweden only)', 'tickera-event-ticketing-system' ),
                'th_TH' => __( 'Thai (for Thailand only)', 'tickera-event-ticketing-system' ),
                'tr_TR' => __( 'Turkish (for Turkey only)', 'tickera-event-ticketing-system' ),
                'zh_CN' => __( 'Simplified Chinese (for China only)', 'tickera-event-ticketing-system' ),
                'zh_HK' => __( 'Traditional Chinese (for Hong Kong only)', 'tickera-event-ticketing-system' ),
                'zh_TW' => __( 'Traditional Chinese (for Taiwan only)', 'tickera-event-ticketing-system' ),
            );

            $this->locales = apply_filters( 'tc_paypal_standard_locales', $locales );
        }

        function payment_form( $cart ) {}

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $order_id = $tc->generate_order_id();

            $params = array();
            $params[ 'no_shipping' ] = '1'; //do not prompt for an address
            $params[ 'cmd' ] = '_xclick';
            $params[ 'business' ] = $this->business;
            $params[ 'currency_code' ] = $this->currency;
            $params[ 'item_name' ] = $this->cart_items();
            $params[ 'amount' ] = $this->total();
            $params[ 'custom' ] = $order_id;
            $params[ 'return' ] = $tc->get_confirmation_slug( true, $order_id );
            $params[ 'cancel_return' ] = apply_filters( 'tc_paypal_standard_cancel_url', $tc->get_cancel_url( $order_id ) );
            $params[ 'notify_url' ] = $this->ipn_url;
            $params[ 'charset' ] = apply_filters( 'tc_paypal_standard_charset', 'UTF-8' );
            $params[ 'rm' ] = '2'; //the buyer's browser is redirected to the return URL by using the POST method, and all payment variables are included
            $params[ 'lc' ] = $this->locale;
            $params[ 'email' ] = $this->buyer_info( 'email' );
            $params[ 'first_name' ] = $this->buyer_info( 'first_name' );
            $params[ 'last_name' ] = $this->buyer_info( 'last_name' );
            $params[ 'bn' ] = 'Tickera_SP';

            if ( 'sandbox' == $this->sandbox_flag ) {
                $params[ 'demo' ] = 'Y';
            }

            $param_list =[];
            foreach ( $params as $k => $v ) {
                $param_list[] = "{$k}=" . rawurlencode( $v );
            }
            $param_str = implode( '&', $param_list );

            $paid = false;
            $payment_info = $this->save_payment_info();
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
            tickera_redirect( "{$this->gateway_url}?{$param_str}", true, false );
            exit( 0 );
        }

        /**
         * Verify Paypal payment.
         * Set Order statuses base on response result
         *
         * @param $order_title
         * @param string $payment_info
         * @param string $cart_info
         */
        function order_confirmation( $order_title, $payment_info = '', $cart_info = '' ) {

            global $tc;

            if ( isset( $_REQUEST[ 'payment_status' ] ) || isset( $_REQUEST[ 'txn_type' ] ) ) {

                $myPost = array();
                $order = tickera_get_order_id_by_name( $order_title );
                $order_id = $order->ID;
                $raw_post_data = file_get_contents( 'php://input' );
                $raw_post_array = explode( '&', $raw_post_data );

                foreach ( $raw_post_array as $keyval ) {
                    $keyval = explode( '=', $keyval );
                    if ( 2 == count( $keyval ) )
                        $myPost[ $keyval[ 0 ] ] = rawurldecode( $keyval[ 1 ] );
                }

                $req = 'cmd=_notify-validate';

                $get_magic_quotes_exists = false;
                if ( defined( 'PHP_VERSION' ) || function_exists( 'phpversion' ) ) {
                    if ( version_compare( PHP_VERSION, '5.4.0', '<' ) ) {
                        $get_magic_quotes_exists = get_magic_quotes_gpc();
                    }
                }

                foreach ( $myPost as $key => $value ) {

                    $value = ( 1 == $get_magic_quotes_exists )
                        ? rawurlencode( stripslashes( $value ) )
                        : rawurlencode( $value );

                    $req .= "&$key=$value";
                }

                $args[ 'user-agent' ] = $tc->title;
                $args[ 'body' ] = $req;
                $args[ 'sslverify' ] = false;
                $args[ 'timeout' ] = 120;

                $response = wp_remote_post( $this->gateway_url, $args );

                if ( 'no' == $this->ignore_ipn_errors ) {

                    if ( is_wp_error( $response ) ) {
                        \Tickera\TC_Order::add_order_note( $order_id, 'PayPal error: ' . sanitize_text_field( $response->get_error_message() ) . '. That means that your website cannot communicate with the PayPal IPN server for some reason. You can turn off IPN check errors by setting option "Ignore IPN errors" to yes or you can contact support.' );
                    }

                    if ( 200 != wp_remote_retrieve_response_code( $response ) ) {
                        \Tickera\TC_Order::add_order_note( $order_id, 'PayPal IPN server responded with a code: ' . sanitize_text_field( wp_remote_retrieve_response_code( $response ) ) . '. That means that your website cannot communicate with the PayPal IPN server for some reason. You can turn off IPN check errors by setting option "Ignore IPN errors" to yes or you can contact support.' );
                    }

                    // Do nothing, wait for IPN message
                }

                /*
                 * Executes if one of the following conditions is met:
                 * Ignore_ipn_errors is set to yes.
                 * Response body is verified.
                 */
                if ( 'yes' == $this->ignore_ipn_errors
                    || ( is_array( $response ) && isset( $response[ 'body' ] ) && 'VERIFIED' == $response[ 'body' ] ) ) {

                    switch ( $_REQUEST[ 'payment_status' ] ) {

                        case 'Completed':
                            $tc->update_order_payment_status( $order_id, true );
                            break;

                        case 'Pending':

                            if ( isset( $_REQUEST[ 'pending_reason' ] ) ) {

                                switch ( $_REQUEST[ 'pending_reason' ] ) {

                                    case 'multi_currency':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'You do not have a balance in the currency sent, and you do not have your profiles\'s Payment Receiving Preferences option set to automatically convert and accept this payment. As a result, you must manually accept or deny this payment. Read more <a href="https://tickera.com/tickera-documentation/settings/payment-gateways/paypal-standard/">here</a>.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'address':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because your customer did not include a confirmed shipping address and your Payment Receiving Preferences is set yo allow you to manually accept or deny each of these payments. To change your preference, go to the Preferences section of your Profile.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'authorization':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'PayPal Order Pending reason: You set the payment action to Authorization and have not yet captured funds.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'echeck':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because it was made by an eCheck that has not yet cleared.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'intl':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because you hold a non-U.S. account and do not have a withdrawal mechanism. You must manually accept or deny this payment from your Account Overview.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'order':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'PayPal Order Pending reason: You set the payment action to Order and have not yet captured funds.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'paymentreview':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending while it is reviewed by PayPal for risk.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'regulatory_review':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because PayPal is reviewing it for compliance with government regulations. PayPal will complete this review within 72 hours.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'unilateral':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because it was made to an email address that is not yet registered or confirmed.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'upgrade':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because it was made via credit card and you must upgrade your account to Business or Premier status before you can receive the funds.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'verify':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending because you are not yet verified. You must verify your account before you can accept this payment.', 'tickera-event-ticketing-system' ) );
                                        break;

                                    case 'other':
                                        \Tickera\TC_Order::add_order_note( $order_id, __( 'The payment is pending for an unknown reason, please contact PayPal Customer Service.', 'tickera-event-ticketing-system' ) );
                                        break;
                                }
                            }
                            break;

                        case 'Canceled':
                            \Tickera\TC_Order::add_order_note( $order_id, __( 'Order cancelled by PayPal (IPN)', 'tickera-event-ticketing-system' ) );
                            $tc->update_order_status( $$order_id, 'order_cancelled' );
                            break;

                        default:
                            // Do nothing, wait for IPN message
                    }

                    $tc->remove_order_session_data();
                }
            }
        }

        function gateway_admin_settings( $settings, $visible ) {
            global $tc;
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Paypal Standard Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( "Sell tickets via PayPal standard payment gateway", 'tickera-event-ticketing-system' ); ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'mode' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                'sandbox' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                'live' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                            'description' => __( 'While testing, you should use Sandbox/Test mode with PayPal test credentials. Once you go to production and want to receive payments via PayPal, you should change this to Live.', 'tickera-event-ticketing-system' )
                        ),
                        'email' => array(
                            'title' => __( 'PayPal account email', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'locale' => array(
                            'title' => __( 'Locale', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->locales,
                            'default' => 'US',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'USD',
                        ),
                        'ignore_ipn_errors' => array(
                            'title' => __( 'Ignore IPN errors', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                'yes' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                'no' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 'no',
                            'description' => __( 'If the paid orders are getting stuck with the status "Order Received" selecting Yes might help resolving that problem', 'tickera-event-ticketing-system' )
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        function ipn() {
            if ( isset( $_REQUEST[ 'custom' ] ) ) {
                do_action( 'tc_custom_paypal_ipn_action' );
                $this->order_confirmation( sanitize_text_field( $_REQUEST[ 'custom' ] ) );
            }
        }

        function add_system_info_options() {
            global $tc; ?>
            <tr>
            <td><?php esc_html_e( 'TLS Version', 'tickera-event-ticketing-system' ); ?>:</td>
            <td class="help"><?php echo wp_kses_post( tickera_tooltip( __( 'The version of TLS.', 'tickera-event-ticketing-system' ) ) ); ?></td>
            <td><?php
                if ( function_exists( 'curl_version' ) ) {
                    $ch = @curl_init();
                    @curl_setopt( $ch, CURLOPT_URL, 'https://www.howsmyssl.com/a/check' );
                    @curl_setopt( $ch, CURLOPT_POST, true );
                    @curl_setopt( $ch, CURLOPT_POSTFIELDS, $request_string );
                    @curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
                    @curl_setopt( $ch, CURLOPT_HEADER, false );
                    @curl_setopt( $ch, CURLOPT_TIMEOUT, 30 );
                    @curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, false );
                    @curl_setopt( $ch, CURLOPT_SSL_VERIFYHOST, false );
                    @curl_setopt( $ch, CURLOPT_VERBOSE, true );
                    $result = @curl_exec( $ch );
                    if ( ! $result ) {
                        esc_html_e( 'N/A', 'tickera-event-ticketing-system' );
                    }
                    @curl_close( $ch );

                    if ( $result ) {
                        $json = json_decode( $result );
                        if ( isset( $json->tls_version ) ) {
                            $tls_version = str_replace( 'TLS ', '', $json->tls_version );
                            if ( (float) $tls_version >= 1.2 ) {
                                echo esc_html( (float) $tls_version );
                            } else {
                                echo wp_kses_post( '<mark class="error"><span class="dashicons dashicons-warning"></span> ' . esc_html( sprintf( /* translators: %s: Paypal Standard TLS Version */ __( '%s - PayPal requires a minimum TLS version of 1.2. We suggest you to contact your hosting and request an update.', 'tickera-event-ticketing-system' ), esc_html( $tls_version ) ) ) . '</mark>' );
                            }
                        } else {
                            esc_html_e( 'N/A', 'tickera-event-ticketing-system' );
                        }
                    }
                }
                ?></td>
            <?php
            $posting = array();

            // fsockopen/cURL.
            $posting[ 'fsockopen_curl' ][ 'name' ] = 'fsockopen/cURL';
            $posting[ 'fsockopen_curl' ][ 'help' ] = wp_kses_post( tickera_tooltip( sprintf( /* translators: %s: Tickera */ __( 'Payment gateways can use cURL to communicate with remote servers to authorize payments, other add-ons may also use it when communicating with remote services. %s use it for ticket templates when getting images. If you have issues with ticket template (blank page), you should turn this option on.', 'tickera-event-ticketing-system' ), $tc->title ), false ) );

            if ( function_exists( 'fsockopen' ) || function_exists( 'curl_init' ) ) {
                $posting[ 'fsockopen_curl' ][ 'success' ] = true;
            } else {
                $posting[ 'fsockopen_curl' ][ 'success' ] = false;
                $posting[ 'fsockopen_curl' ][ 'note' ] = __( 'Your server does not have fsockopen or cURL enabled - PayPal IPN and other scripts which communicate with other servers will not work. Contact your hosting provider.', 'tickera-event-ticketing-system' );
            }

            // WP Remote Post Check.
            $posting[ 'wp_remote_post' ][ 'name' ] = __( 'Remote Post', 'tickera-event-ticketing-system' );
            $posting[ 'wp_remote_post' ][ 'help' ] = wp_kses_post( tickera_tooltip( __( 'PayPal uses this method of communicating when sending back transaction information.', 'tickera-event-ticketing-system' ), false ) );

            $response = wp_safe_remote_post( 'https://www.paypal.com/cgi-bin/webscr', array(
                'timeout' => 60,
                'user-agent' => $tc->title . '/' . $tc->version,
                'httpversion' => '1.1',
                'body' => array(
                    'cmd' => '_notify-validate'
                )
            ) );

            if ( ! is_wp_error( $response ) && $response[ 'response' ][ 'code' ] >= 200 && $response[ 'response' ][ 'code' ] < 300 ) {
                $posting[ 'wp_remote_post' ][ 'success' ] = true;
            } else {
                $posting[ 'wp_remote_post' ][ 'success' ] = false;
            }

            // WP Remote Get Check.
            $posting[ 'wp_remote_get' ][ 'name' ] = __( 'Remote Get', 'tickera-event-ticketing-system' );
            $posting[ 'wp_remote_get' ][ 'help' ] = wp_kses_post( tickera_tooltip( sprintf( /* translators: %s: Tickera */ __( '%s plugin and/or its add-ons may use this when checking for plugin updates.', 'tickera-event-ticketing-system' ), $tc->title ), false ) );

            $response = wp_safe_remote_get( 'https://www.paypal.com/cgi-bin/webscr', array(
                'timeout' => 60,
                'user-agent' => $tc->title . '/' . $tc->version,
                'httpversion' => '1.1',
                'body' => array(
                    'cmd' => '_notify-validate'
                )
            ) );

            if ( ! is_wp_error( $response ) && $response[ 'response' ][ 'code' ] >= 200 && $response[ 'response' ][ 'code' ] < 300 ) {
                $posting[ 'wp_remote_get' ][ 'success' ] = true;
            } else {
                $posting[ 'wp_remote_get' ][ 'success' ] = false;
            }

            foreach ( $posting as $post ) {
                $mark = ! empty( $post[ 'success' ] ) ? 'yes' : 'error';
                ?>
                <tr>
                    <td><?php echo esc_html( $post[ 'name' ] ); ?>:</td>
                    <td class="help"><?php echo esc_html( isset( $post[ 'help' ] ) ? $post[ 'help' ] : '' ); ?></td>
                    <td>
                        <mark class="<?php echo esc_attr( $mark ); ?>">
                            <?php echo wp_kses_post( ! empty( $post[ 'success' ] ) ? '<span class="dashicons dashicons-yes"></span>' : '<span class="dashicons dashicons-no-alt"></span>' ); ?><?php echo wp_kses_post( ! empty( $post[ 'note' ] ) ? wp_kses_post( $post[ 'note' ] ) : '' ); ?>
                        </mark>
                    </td>
                </tr>
                <?php
            }
            ?>
            </tr>
            <?php
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_PayPal_Standard', 'paypal_standard', __( 'PayPal Standard', 'tickera-event-ticketing-system' ) );
}