<?php
/**
 * PayU - Payment Gateway (API: http://developers.payulatam.com/en/web_checkout/integration.html)
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_PayU_Latam' ) ) {

    class TC_Gateway_PayU_Latam extends TC_Gateway_API {

        var $plugin_name = 'payu_latam';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;
        var $API_Username, $API_Password, $mode, $returnURL, $API_Endpoint, $version, $locale;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = true;
        var $currency;
        var $merchant_id;
        var $account_id;
        var $api_login;
        var $api_key;
        var $language;
        var $live_url;
        var $test_url;
        var $languages;
        var $hourly_interval;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
            add_action( 'tc_save_tc_gateway_settings', array( $this, 'schedule_order_status_confirmation' ) );
            add_action( 'tc_payu_latam_maybe_confirm_pending_posts_hook', array( $this, 'maybe_confirm_pending_orders' ) );
        }

        function init() {
            global $tc;

            $this->admin_name = __( 'PayU Latam', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'PayU Latam', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/payulatam.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-payulatam.png', $this->plugin_name );

            $this->currency = $this->get_option( 'currency', 'ARS' );
            $this->mode = $this->get_option( 'mode', 'sandbox' );
            $this->merchant_id = $this->get_option( 'merchant_id' );
            $this->account_id = $this->get_option( 'account_id' );
            $this->api_login = $this->get_option( 'api_login' );
            $this->api_key = $this->get_option( 'api_key' );
            $this->language = $this->get_option( 'language', 'EN' );

            $this->live_url = 'https://checkout.payulatam.com/ppp-web-gateway-payu/';//'https://gateway.payulatam.com/ppp-web-gateway/';
            $this->test_url = 'https://sandbox.checkout.payulatam.com/ppp-web-gateway-payu/';//'https://stg.gateway.payulatam.com/ppp-web-gateway';

            $this->currencies = [
                "ARS" => __( 'ARS - Argentine Peso', 'tickera-event-ticketing-system' ),
                "BRL" => __( 'BRL - Brazilian Real', 'tickera-event-ticketing-system' ),
                "CLP" => __( 'CLP - Chilean Peso', 'tickera-event-ticketing-system' ),
                "COP" => __( 'COP - Colombian Peso', 'tickera-event-ticketing-system' ),
                "MXN" => __( 'MXN - Mexican Peso', 'tickera-event-ticketing-system' ),
                "PEN" => __( 'PEN - Peruvian Nuevo Sol', 'tickera-event-ticketing-system' ),
                "USD" => __( 'USD - US Dollar', 'tickera-event-ticketing-system' ),
            ];

            $this->languages = [
                'ES' => 'Spanish',
                'EN' => 'English',
                'PT' => 'Portuguese'
            ];

            for ( $i = 1; $i <= 72; $i++ ) {
                $this->hourly_interval[ $i ] = sprintf( /* translators: %d: Iteration from 1 to 72 */ _n( 'Every %d hour', 'Every %d hours', $i, 'tickera-event-ticketing-system' ), $i );
            }
        }

        function payment_form( $cart ) {
            global $tc;
        }

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $url = ( 'sandbox' == $this->mode ) ? $this->test_url : $this->live_url;

            $this->save_cart_info();
            $order_id = $tc->generate_order_id();
            $paid = false;
            $payment_info = $this->save_payment_info();
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
            $str = $this->api_key . "~" . $this->merchant_id . "~" . $order_id . "~" . $this->total() . "~" . $this->currency;
            $hash = strtolower( md5( $str ) );

            $payulatam_args = array(
                'merchantId' => $this->merchant_id,
                'accountId' => $this->account_id,
                'signature' => $hash,
                'referenceCode' => $order_id,
                'amount' => $this->total(),
                'currency' => $this->currency,
                'payerFullName' => $this->buyer_info( 'full_name' ),
                'buyerEmail' => $this->buyer_info( 'email' ),
                'lng' => $this->language,
                'description' => $this->cart_items(),
                'responseUrl' => $tc->get_confirmation_slug( true, $order_id ),
                'confirmationUrl' => $tc->get_confirmation_slug( true, $order_id ),
                'tax' => 0,
                'taxReturnBase' => 0,
                'extra1' => $order_id,
                'discount' => '0',
            );

            if ( $this->mode == 'sandbox' ) {
                $payulatam_args[ 'test' ] = '1';
            }

            $payulatam_args_array = array();

            foreach ( $payulatam_args as $key => $value ) {
                $payulatam_args_array[] = '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $value ) . '" />';
            }

            header( 'Content-Type: text/html' ); ?>
            <form action="<?php echo esc_url( $url ) ?>" method="POST" id="payulatam_payment_form" target="_top"><?php echo implode( '', $payulatam_args_array ) ?><script type="text/javascript">document.getElementById("payulatam_payment_form").submit();</script></form>' );<?php
        }

        function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {
            $request = tickera_sanitize_array( $_REQUEST, false, true );
            $this->request( $request );
        }

        function gateway_admin_settings( $settings, $visible ) { ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: PayU Latam Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( 'Sell your tickets via PayUlatam payment gateway', 'tickera-event-ticketing-system' ); ?></span>
                </h3>
                <div class="inside">
                    <?php
                    $fields = array(
                        'mode' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                'sandbox' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                'live' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 'sandbox',
                        ),
                        'merchant_id' => array(
                            'title' => __( 'Merchant ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'account_id' => array(
                            'title' => __( 'Account ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'api_login' => array(
                            'title' => __( 'Api Login', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'api_key' => array(
                            'title' => __( 'Api key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'language' => array(
                            'title' => __( 'Gateway language', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->languages,
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                        ),
                        'periodically_confirm_order_status' => array(
                            'title' => __( 'Periodically confirm order status', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => [
                                'yes' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                'no' => __( 'No', 'tickera-event-ticketing-system' )
                            ],
                            'default' => 'no',
                            'description' => __( 'There\'s an instance when a payment gateway fails to return transaction response to the merchant website. This setting will allow the merchant\'s website to periodically confirms order status.', 'tickera-event-ticketing-system' )
                        ),
                        'periodically_confirm_order_status_interval' => array(
                            'title' => __( 'Periodically confirm order status interval', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->hourly_interval,
                            'default' => 24,
                            'conditional' => array(
                                'field_name' => 'periodically_confirm_order_status',
                                'field_type' => 'select',
                                'value' => 'yes',
                                'action' => 'show'
                            )
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        function request( $posted ) {

            global $tc;

            if ( ! empty( $posted[ 'referenceCode' ] ) ) {

                $order = tickera_get_order_id_by_name( $posted[ 'referenceCode' ] );

                if ( ! empty( $posted[ 'transactionState' ] ) ) {

                    // We are here so lets check status and do actions
                    if ( isset( $posted[ 'transactionState' ] ) && in_array( $posted[ 'transactionState' ], [ '4', 'APPROVED' ] ) ) { // 4 = APPROVED
                        $tc->update_order_payment_status( $order->ID, true );
                        \Tickera\TC_Order::add_order_note( $order->ID, __( 'Transaction approved.', 'tickera-event-ticketing-system' ) );

                    } elseif ( isset( $posted[ 'transactionState' ] ) && in_array( $posted[ 'transactionState' ], [ '6', 'REJECTED', 'DECLINED' ] ) ) { // 6 = REJECTED
                        \Tickera\TC_Order::add_order_note( $order->ID, __( 'Transaction rejected.', 'tickera-event-ticketing-system' ) );

                    } elseif ( isset( $posted[ 'transactionState' ] ) && in_array( $posted[ 'transactionState' ], [ '104', 'ERROR' ] ) ) { // 104 = ERROR
                        \Tickera\TC_Order::add_order_note( $order->ID, __( 'Transaction error.', 'tickera-event-ticketing-system' ) );

                    } elseif ( isset( $posted[ 'transactionState' ] ) && in_array( $posted[ 'transactionState' ], [ '7', 'PENDING' ] ) ) { // 7 = PENDING
                        \Tickera\TC_Order::add_order_note( $order->ID, __( 'Pending payment...', 'tickera-event-ticketing-system' ) );
                    }
                }

            } else {
                $this->confirmation( $posted );
            }
        }

        function confirmation( $posted ) {
            global $tc;

            if ( isset( $posted[ 'reference_sale' ] ) && ! empty( $posted[ 'reference_sale' ] ) ) {

                $order = tickera_get_order_id_by_name( $posted[ 'reference_sale' ] );

                if ( ! empty( $posted[ 'response_message_pol' ] ) ) {

                    // We are here so lets check status and do actions
                    if ( isset( $posted[ 'response_message_pol' ] ) && $posted[ 'response_message_pol' ] == 'APPROVED' ) {
                        $tc->update_order_payment_status( $order->ID, true );
                    }
                }
            }
        }

        function schedule_order_status_confirmation() {

            $periodically_confirm_order_status = $this->get_option( 'periodically_confirm_order_status', 'no' );

            if ( 'yes' == $periodically_confirm_order_status ) {

                if ( ! wp_next_scheduled( 'tc_payu_latam_maybe_confirm_pending_posts_hook' ) ) {
                    wp_schedule_event( time(), 'hourly', 'tc_payu_latam_maybe_confirm_pending_posts_hook' );
                }

                self::maybe_confirm_pending_orders();

            } else {
                wp_clear_scheduled_hook( 'tc_payu_latam_maybe_confirm_pending_posts_hook' );
            }
        }

        function maybe_confirm_pending_orders() {

            global $wpdb;

            $periodically_confirm_order_status = $this->get_option( 'periodically_confirm_order_status', 'no' );
            $periodically_confirm_order_status_interval = $this->get_option( 'periodically_confirm_order_status_interval', 24 );

            if ( 'yes' == $periodically_confirm_order_status ) {

                $pending_orders = $wpdb->get_results( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_date < (NOW() - INTERVAL %d HOUR) AND post_type = 'tc_orders' AND post_status = 'order_received'", (int) $periodically_confirm_order_status_interval ), OBJECT );

                foreach ( $pending_orders as $pending_order ) {

                    $order_id = $pending_order->ID;
                    $order = new \Tickera\TC_Order( $order_id );

                    if ( 'TC_Gateway_PayU_Latam' == $order->details->tc_cart_info[ 'gateway_class' ] ) {

                        $confirmation_response = self::api( [
                            'test' => ( 'sandbox' == $this->mode ) ? true : false,
                            'language' => strtolower( $this->language ),
                            'command' => 'ORDER_DETAIL_BY_REFERENCE_CODE',
                            'merchant' => [
                                'apiLogin' => $this->api_login,
                                'apiKey' => $this->api_key
                            ],
                            'details' => [
                                'referenceCode' => $order->details->post_title
                            ]
                        ]);

                        if ( $confirmation_response
                            && isset( $confirmation_response[ 'code' ] ) && 'SUCCESS' == $confirmation_response[ 'code' ]
                            && isset( $confirmation_response[ 'result' ] ) && isset( $confirmation_response[ 'result' ][ 'payload' ] )
                            && isset( $confirmation_response[ 'result' ][ 'payload' ] )
                            && isset( $confirmation_response[ 'result' ][ 'payload' ][ 'order' ][ 'transactions' ][ 'transaction' ][ 'transactionResponse' ][ 'state' ] )
                        ) {

                            $order = $confirmation_response[ 'result' ][ 'payload' ][ 'order' ];
                            $transaction = $confirmation_response[ 'result' ][ 'payload' ][ 'order' ][ 'transactions' ][ 'transaction' ];

                            $posted = [
                                'referenceCode' => $order[ 'referenceCode' ],
                                'transactionState' => $transaction[ 'transactionResponse' ][ 'state' ],
                            ];

                            self::request( $posted );
                        }
                    }
                }
            }
        }

        function api( $data ) {

            $url = ( 'sandbox' == $this->mode )
                ? 'https://sandbox.api.payulatam.com/reports-api/4.0/service.cgi'
                : 'https://api.payulatam.com/reports-api/4.0/service.cgi';

            $response = wp_remote_post( $url, array (
                'method' => 'POST',
                'timeout' => 45,
                'redirection' => 5,
                'httpversion' => '1.0',
                'blocking' => true,
                'headers' => [ 'content-type' => 'application/json' ],
                'body' => json_encode( $data ),
                'cookies' => [],
                'sslverify' => true
            ) );

            if ( ! is_wp_error( $response ) ) {
                $response = simplexml_load_string( wp_remote_retrieve_body( $response ) );
                $response = json_encode( $response );
                return json_decode( $response, true );

            } else {
                return false;
            }
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_PayU_Latam', 'payu_latam', __( 'PayU Latam', 'tickera-event-ticketing-system' ) );
}